#pragma rtGlobals=3		// Use modern global access method and strict wave access.

Menu "Load Waves"
	"Load PAM File...", LoadPAMFile("", "", "PAMDateAndTime", "PAMDataMatrix", 1)
	"Load All PAM Files in Folder...", LoadAllPAMFilesInFolder()
End

// AMPMTimeAdjustment(theTime, flag)
// Returns the necessary adjustment in seconds to convert a time from AM/PM representation
// to 24 hour representation.
Function AMPMTimeAdjustment(theTime, flag)
	Variable theTime
	String flag					// Must be either "a" or "p"
	
	Variable adjustment = 0
	Variable secondsIn12Hours = 43200
	
	if (CmpStr(flag, "a") == 0)
		// AM: If the hour is 12:xx, subtract seconds in 12 hours
		if (theTime >= secondsIn12Hours)
			adjustment = -secondsIn12Hours
		endif
	else
		// PM: Adds 12 hours unless hour is 12:xx.
		if (theTime < secondsIn12Hours)
			adjustment = secondsIn12Hours
		endif
	endif

	return adjustment
End

Function LoadPAMFile(pathName, fileName, dateTimeWaveName, dataMatrixName, showInTable)
	String pathName			// Name of an Igor symbolic path or ""
	String fileName				// Name of file or full path to file
	String dateTimeWaveName	// Name to use for date/time wave
	String dataMatrixName		// Name to use for data wave
	Variable showInTable
 
	// First get a valid reference to a file.
	if ((strlen(pathName)==0) || (strlen(fileName)==0))
		// Display dialog looking for file.
		Variable refNum
		Open/D/R/F="*.txt"/P=$pathName refNum as fileName
		fileName = S_fileName			// S_fileName is set by Open/D
		if (strlen(fileName) == 0)		// User cancelled?
			return -1
		endif
	endif

	String columnInfoStr
	String delimiters
	
	// First load the date/time column
	delimiters = "\t"
	columnInfoStr = "N='" + dateTimeWaveName + "',F=8;"		// Load as date/time
	LoadWave /J /Q /L={0,1,0,0,1} /V={delimiters,"",0,0} /A /O /B=columnInfoStr /R={English, 1, 2, 1, 1, "Month/DayofMonth/Year", 40} /P=$pathName fileName
	Wave DateAndTime = $dateTimeWaveName
	
	// Next load the AM/PM column
	
	delimiters = "\t" + " "							// Need space so that the AM/PM column will be taken as distinct from the preceding time column
	columnInfoStr = "N='Date',F=6;"					// Skip first column (date)
	columnInfoStr += "N='Time',F=7;"				// Skip first column (time)
	columnInfoStr += "N=TempAMPM,F=-2;"			// Load as text
	LoadWave /J /Q /L={0,1,0,2,1} /V={delimiters,"",0,0} /A /O /B=columnInfoStr /P=$pathName fileName
	Wave/T TempAMPM
	
	// Account for AM/PM affect on time
	DateAndTime += AMPMTimeAdjustment(DateAndTime, TempAMPM)
	KillWaves/Z TempAMPM							// This is no longer needed
	
	// Finally load the rest of the columns as a matrix
	delimiters = "\t"									// We will treat everything up to the first tab as text to easily skip it
	columnInfoStr = "N='_skip_',F=-2;"				// Treat as text
	columnInfoStr += "N='" + dataMatrixName + "',F=0;"		// Load as numeric
	LoadWave /J /Q /M /L={0,1,0,1,0} /V={delimiters,"",0,0} /A /O /B=columnInfoStr /P=$pathName fileName
	Wave dataMatrix = $dataMatrixName
	
	Printf "Loaded PAM data from \"%s\"\r", fileName
	
	if (showInTable)
		Edit DateAndTime, dataMatrix
		ModifyTable format(DateAndTime)=8, width(DateAndTime)=160
	endif

	return 0							// Success
End

// LoadAllPAMFiles()
// Loads PAM data into the current data folder.
// The symbolic path PAMDataPath is expected to exist.
// Creates waves named PamDateAndTime and PamDataMatrix, overwriting any pre-existing waves with the same names.
Function LoadAllPAMFiles()
	// Counts number of files in folder	
	String fileList = IndexedFile(PAMDataPath, -1, ".txt")		
	Variable NumFiles = ItemsInList(fileList)
	
	Variable numFilesLoaded = 0
	Variable fileNum
	For(fileNum=0; fileNum<numfiles; fileNum+=1)		// loops through each file name	
		String fileName =StringFromList(fileNum, fileList)	// get full path to data file	, specify extensions e.g. ".txt"
		if (StringMatch(fileName,"*PAM*") == 0)
			continue
		endif
		
		Variable refNum
		Open /R/P=PAMDataPath/Z refNum as fileName	// open
		FStatus refNum	// check there is data in file
		Variable numBytesInFile = V_logEOF
		Close refNum

		if (numBytesInFile < 10)	// specify minimum file size to load e.g. >10bytes
			continue
		endif
		
		LoadPAMFile("PAMDataPath", fileName, "tempDateAndTime", "tempDataMatrix", 0)	// Creates tempDateAndTime and tempDataMatrix waves
		Wave tempDateAndTime
		Wave tempDataMatrix

		// Append loaded waves to previous data
		if (numFilesLoaded == 0)
			Duplicate/O tempDateAndTime, PAMDateAndTime
			Duplicate/O tempDataMatrix, PAMDataMatrix
			KillWaves/Z tempDateAndTime, tempDataMatrix
		else
			Wave PAMDateAndTime
			Concatenate /KILL /NP {tempDateAndTime}, PAMDateAndTime
			Wave PAMDataMatrix
			Concatenate /KILL /NP=0 {tempDataMatrix}, PAMDataMatrix
		endif
		
		numFilesLoaded += 1
	EndFor
	
	Variable numRows = DimSize(PAMDataMatrix, 0)
	Printf "Number of files loaded: %d. Total number of rows: %d\r", numFilesLoaded, numRows
End

// LoadAllPAMFilesInFolder()
// Loads PAM data into the current data folder.
// Creates waves named PamDateAndTime and PamDataMatrix, overwriting any pre-existing waves with the same names.
Function LoadAllPAMFilesInFolder()
	// Get folder with data in it
	NewPath/M="Select folder containing PAM time data:"/O/Q PAMDataPath	// prompts user to select path to data folder
	// Abort if cancel is pressed
	if (V_flag!=0)		
		return -1	
	EndIf
	
	LoadAllPAMFiles()
	
	return 0
End
